/**
 * \file: mspin_connection_tcp_client.c
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * mySPIN Connection Adapter TCP/IP
 *
 * \component: MSPIN
 *
 * \author: Thilo Bjoern Fickel BSOT/PJ-ES1 thilo.fickel@bosch-softtec.com
 *
* \copyright: (c) 2003 - 2016 ADIT Corporation
 *
 * \history
 * 0.1 TFickel Initial version
 *
 ***********************************************************************/

#include "mspin_connection_tcp_client.h"
#include "mspin_connection_tcp.h"
#include "mspin_connection_adapter.h"
#include "mspin_connection_tcp_manager.h"
#include "mspin_logging.h"

#include <pthread.h>        //pthread_*
#include <string.h>         //memset
#include <netdb.h>          //getaddrinfo
#include <sys/socket.h>     //socket
#include <netinet/in.h>     //IPPROTO_UDP
#include <netinet/udp.h>    //udp
#include <errno.h>          //errno
#include <sys/prctl.h>      //prctl
#include <arpa/inet.h>
#include <mqueue.h>         //mq_open,...
#include <ifaddrs.h>        //ifaddrs

#define MSPIN_MAX_UDP_PACKET_SIZE 10000
#define MSPIN_UDP_LISTENER_SELECT_TIMEOUT_MS 500

#ifdef MSPIN_TCP_USE_MESSAGE_QUEUE
#define MSPIN_TCP_READ_MAX_NUMBER_MESSAGES 20 //adjustments required here for performance improvements?
#define MSPIN_TCP_MESSAGE_QUEUE "/mspintcpqueue"
#endif //MSPIN_TCP_USE_MESSAGE_QUEUE

typedef struct
{
    int socket;
    MSPIN_UDPMessageReceived msgReceivedCB;
    void* context;
    BOOL quitThread;
    pthread_t threadID;
} mspin_tcp_udpListenerContext_t;

typedef enum
{
    MSPIN_UDP_MESSAGE_PARSED = 0,           // parsing ok
    MSPIN_UDP_MESSAGE_PARSING_FAILED = -1   // parsing failed
} mspin_udp_parsing_result_t;

//Note: Using a static context makes sure that only one UDP listener is started. Otherwise the context could be
// returned within mspin_tcp_startUDPListener and handed over within mspin_tcp_stopUDPListener
static mspin_tcp_udpListenerContext_t *gpUdpListenerContext = NULL;

static mspin_udp_parsing_result_t mspin_tcp_parseUDPMessage(char* message, char* pHostname, unsigned int *pPort)
{
    //A typical message looks like this: 'mySPINd://192.168.43.1:7195'

    //Check parameters
    if (!pPort || !pHostname)
    {
        if (!pPort)
        {
            mspin_log_printLn(eMspinVerbosityError, "%s(mesg='%s') ERROR: Port buffer is NULL",
                    __FUNCTION__, message);
        }

        if (!pHostname)
        {
            mspin_log_printLn(eMspinVerbosityError, "%s(mesg='%s') ERROR: Hostname buffer is NULL",
                    __FUNCTION__, message);
        }

        return MSPIN_UDP_MESSAGE_PARSING_FAILED;
    }

    char* part;
    part = strtok(message, ":/");
    if (!part)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(mesg='%s') ERROR: Nothing found",
                __FUNCTION__, message);
        return MSPIN_UDP_MESSAGE_PARSING_FAILED;
    }

    if (0 != strcmp(part, MSPIN_UDP_MESSAGE_PREFIX))
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(mesg='%s') ERROR: Prefix '%s' not found",
                __FUNCTION__, message, MSPIN_UDP_MESSAGE_PREFIX);
        return MSPIN_UDP_MESSAGE_PARSING_FAILED;
    }

    part = strtok(NULL, ":/");
    if (!part)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(mesg='%s') ERROR: Hostname no found",
                __FUNCTION__, message);
        return MSPIN_UDP_MESSAGE_PARSING_FAILED;
    }

    //Copy hostname
    if (sizeof(pHostname) < strlen(part))
    {
        strncpy(pHostname, part, strlen(part));
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(mesg='%s') ERROR: Hostname buffer is too small",
                __FUNCTION__, message);

        return MSPIN_UDP_MESSAGE_PARSING_FAILED;
    }

    mspin_log_printLn(eMspinVerbosityDebug, "%s(mesg='%s') hostname='%s'  found",
            __FUNCTION__, message, pHostname);

    return MSPIN_UDP_MESSAGE_PARSED;
}

static void* mspin_tcp_listenUDP(void* exinf)
{
    mspin_tcp_udpListenerContext_t* pContext = (mspin_tcp_udpListenerContext_t*)exinf;

    struct sockaddr_in si_other={0};

    fd_set fds = {{0}};
    struct timeval tv;
    int selectResult = 0;

    int bytesReceived = 0;
    char buf[MSPIN_MAX_UDP_PACKET_SIZE];
    unsigned int slen = sizeof(struct sockaddr);
    char hostname[128] = {0};
    unsigned int port = 0;

    //Set thread name
    prctl(PR_SET_NAME, "mspin_UDPListen", 0, 0, 0);

    //Loop till quitting UDP listener thread
    // (and as long as the context and socket fd is valid)
    while(pContext && !pContext->quitThread
            && (pContext->socket >= 0))
    {
        int udpSocket = pContext->socket; //store in local variable which is valid till this iteration
        /* PRQA: Lint Message 529: Warning due symbol not subsequently referenced in the asm section. Not an issue */
        /*lint -save -e529*/
        FD_ZERO(&fds);
        /*lint -restore*/
        FD_SET(udpSocket, &fds);
        tv.tv_sec = MSPIN_UDP_LISTENER_SELECT_TIMEOUT_MS/1000; //seconds
        tv.tv_usec = (MSPIN_UDP_LISTENER_SELECT_TIMEOUT_MS%1000)*1000; //milliseconds

        selectResult = select(udpSocket+1, &fds, NULL, NULL, &tv); //check read fds
        if ((selectResult > 0) && FD_ISSET(udpSocket, &fds))
        {
            /* PRQA: Lint Message 64: accept type mismatch */
            /*lint -save -e64*/
            bytesReceived = recvfrom(udpSocket, buf, sizeof(buf)-1, 0, (struct sockaddr *)&si_other, &slen);
            /*lint -restore*/
            if (-1 == bytesReceived)
            {
                mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Failed to receive bytes with errno=%s (%d)",
                        __FUNCTION__, strerror(errno), errno);

                usleep(500*1000); //in case of error do not try immediately again, wait 500ms
            }
            else if (0 == bytesReceived)
            {
                mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Connection closed", __FUNCTION__);
            }
            else
            {
                if (pContext->msgReceivedCB)
                {
                    mspin_log_printLn(eMspinVerbosityDebug, "%s() %d bytes received from IP='%s' with mesg='%s'",
                            __FUNCTION__, bytesReceived, inet_ntoa(si_other.sin_addr), buf);

                    //Parse message
                    if (MSPIN_UDP_MESSAGE_PARSED == mspin_tcp_parseUDPMessage(buf, hostname, &port))
                    {
                        mspin_log_printLn(eMspinVerbosityError, "%s() hostname='%s' and port=%d parsed",
                                __FUNCTION__, hostname, port);

                        //Trigger callback
                        pContext->msgReceivedCB((const U8*)hostname, port, pContext->context);
                    }
                    else
                    {
                        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: parsing of '%s' failed",
                                __FUNCTION__, buf);
                    }
                }
                else
                {
                    mspin_log_printLn(eMspinVerbosityError, "%s() %d bytes received from IP='%s' with mesg='%s' but callback is NULL",
                            __FUNCTION__, bytesReceived, inet_ntoa(si_other.sin_addr), buf);
                }
            }
        }
        else if (selectResult < 0)
        {
            mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: select() returned with error code=%d and errno='%s'(%d) => abort",
                    __FUNCTION__, selectResult, strerror(errno), errno);
            break;
        }
        else if (selectResult == 0)
        {
            //Timeout, nothing to do
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: select() indicated data on invalid fd",
                    __FUNCTION__, selectResult);
        }
    }

    mspin_log_printLn(eMspinVerbosityDebug, "%s() terminate UDP listener thread",
            __FUNCTION__);

    pthread_exit((void*)exinf);
    return NULL;
}

static S32 mspin_tcp_startUDPListenerThread(mspin_tcp_udpListenerContext_t* context)
{
    pthread_attr_t attr;
    S32 rc = -1;

    memset(&attr, 0, sizeof(pthread_attr_t));

    rc = pthread_attr_init(&attr);
    if (0 != rc)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Failed to init thread attributes", __FUNCTION__);
        return rc;
    }

    rc = pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_JOINABLE);
    if (0 != rc)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Failed to set detach state", __FUNCTION__);
        return rc;
    }

    context->quitThread = FALSE;

    rc = pthread_create(&context->threadID, &attr, mspin_tcp_listenUDP, (void*)context);
    if (0 == rc)
    {
        mspin_log_printLn(eMspinVerbosityDebug, "%s() using thread id=%d", __FUNCTION__, context->threadID);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Failed to create thread", __FUNCTION__);
        context->threadID = 0;
    }

    (void)pthread_attr_destroy(&attr);

    return rc;
}

static S32 mspin_tcp_openSocket(S32 port)
{
    struct sockaddr_in si_me;
    int udpSocket = 0;
    int broadcast = 1;

    udpSocket = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);

    if (-1 == udpSocket)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Failed to create socket with errno=%s (%d)",
                __FUNCTION__, strerror(errno), errno);

        return -1;
    }

    if(-1 == setsockopt(udpSocket, SOL_SOCKET, SO_BROADCAST, &broadcast, sizeof broadcast))
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Failed to set socket options with errno=%s (%d)",
                __FUNCTION__, strerror(errno), errno);

        close(udpSocket);
        return -2;
    }

    memset(&si_me, 0, sizeof(si_me));
    si_me.sin_family = AF_INET;
    si_me.sin_port = htons(port);
    si_me.sin_addr.s_addr = INADDR_ANY;
    /* PRQA: Lint Message 64: accept type mismatch */
    /*lint -save -e64*/
    if (-1 == bind(udpSocket, (struct sockaddr *)&si_me, sizeof(struct sockaddr)))
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Failed to bind socket with errno=%s (%d)",
                __FUNCTION__, strerror(errno), errno);

        close(udpSocket);
        return -3;
    }
    /*lint -restore*/
    mspin_log_printLn(eMspinVerbosityDebug, "%s() Socket FD=%d created and successfully bound", __FUNCTION__, udpSocket);

    return udpSocket;
}

MSPIN_ERROR mspin_tcp_startUDPListener(S32 port, MSPIN_UDPMessageReceived mesgReceivedCB, void *pUDPListenerContext)
{
    if (!gpUdpListenerContext)
    {
        gpUdpListenerContext = malloc(sizeof(mspin_tcp_udpListenerContext_t));

        if (!gpUdpListenerContext)
        {
            mspin_log_printLn(eMspinVerbosityError, "%s(port=%d, udpCtx=%p) ERROR: Failed to allocate memory for UDP Listener context",
                    __FUNCTION__, port, pUDPListenerContext);
            return MSPIN_ERROR_MEM_ALLOC;
        }

        //Initialize UDP listener structure
        memset(gpUdpListenerContext, 0, sizeof(mspin_tcp_udpListenerContext_t));
        gpUdpListenerContext->socket = -1; //initialize socket FD with -1

        //Open socket and bind it before creating thread in order to check if opening succeeds
        gpUdpListenerContext->socket = mspin_tcp_openSocket(port);
        if (gpUdpListenerContext->socket < 0)
        {
            mspin_log_printLn(eMspinVerbosityError, "%s(port=%d, udpCtx=%p) ERROR: Failed to open socket",
                    __FUNCTION__, port, pUDPListenerContext);

            free(gpUdpListenerContext);
            gpUdpListenerContext = NULL;

            return MSPIN_ERROR_GENERAL;
        }

        if (mspin_tcp_startUDPListenerThread(gpUdpListenerContext) == -1)
        {
            mspin_log_printLn(eMspinVerbosityError, "%s(port=%d, udpCtx=%p) ERROR: Failed to start UDP listener",
                    __FUNCTION__, port, pUDPListenerContext);

            free(gpUdpListenerContext);
            gpUdpListenerContext = NULL;

            return MSPIN_ERROR_GENERAL;
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityDebug, "%s(port=%d, udpCtx=%p) UDP listener started",
                    __FUNCTION__, port, pUDPListenerContext);

            gpUdpListenerContext->msgReceivedCB = mesgReceivedCB;
            gpUdpListenerContext->context = pUDPListenerContext;

            return MSPIN_SUCCESS;
        }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(udpCtx=%p) ERROR: Another UDP listener is already registered. De-register first",
                __FUNCTION__, pUDPListenerContext);
        return MSPIN_ERROR_NOT_READY;
    }
}

MSPIN_ERROR mspin_tcp_stopUDPListener()
{
    void* status;

    if (gpUdpListenerContext)
    {
        //De-register callback
        gpUdpListenerContext->msgReceivedCB = NULL;
        gpUdpListenerContext->context = NULL;

        //Join UDP listener
        gpUdpListenerContext->quitThread = TRUE;

        if (gpUdpListenerContext->threadID > 0)
        {
            //do not use pthread_cancel() because in this case the socket would not be closed and could not be reused
            //(void)pthread_cancel(gUDPListenerThreadID);

            mspin_log_printLn(eMspinVerbosityDebug, "%s() join thread", __FUNCTION__);

            (void)pthread_join(gpUdpListenerContext->threadID, &status);
            gpUdpListenerContext->threadID = 0;
            gpUdpListenerContext->quitThread = FALSE; //reset already here to make sure it is reset

            mspin_log_printLn(eMspinVerbosityDebug, "%s() UDP listener thread joined", __FUNCTION__);
        }

        //Close socket
        close(gpUdpListenerContext->socket);
        gpUdpListenerContext->socket = -1;

        //Free context
        free(gpUdpListenerContext);
        gpUdpListenerContext = NULL;

        mspin_log_printLn(eMspinVerbosityDebug, "%s() socket closed and context freed", __FUNCTION__);

        return MSPIN_SUCCESS;
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: No UDP listener context", __FUNCTION__);
        return MSPIN_ERROR_GENERAL;
    }
}

MSPIN_ERROR mspin_tcp_connectServer(mspin_context_t* pContext, const U8* ipAddr, unsigned int port)
{
    mspin_connectionHandle_t *pConnectionHandle = NULL;
    struct sockaddr_in si_host = {0};
    struct hostent *host = NULL;
    int socketFD = -1;
    MSPIN_ERROR result = MSPIN_SUCCESS;

    mspin_log_printLn(eMspinVerbosityDebug, "%s(ctx=%p, ip='%s', port=%d) entered", __FUNCTION__, pContext, ipAddr, port);

    if (pContext)
    {
        if (!ipAddr)
        {
            mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p, ip='%s', port=%d) ERROR: ipAddr is NULL",
                    __FUNCTION__, pContext, ipAddr, port);
            return MSPIN_ERROR_INVALID_PARAMETER;
        }

        host = gethostbyname((const char*)ipAddr);
        memcpy(&(si_host.sin_addr), host->h_addr,host->h_length);
        si_host.sin_family = host->h_addrtype;
        si_host.sin_port = htons(port);

        socketFD = socket(AF_INET, SOCK_STREAM, 0);
        if (-1 == socketFD)
        {
            mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p, ip='%s', port=%d) ERROR: Creating socket failed with '%s'(%d)",
                    __FUNCTION__, pContext, ipAddr, port, strerror(errno), errno);
            return MSPIN_ERROR_CONNECTION_START;
        }
        /* PRQA: Lint Message 64: accept type mismatch */
        /*lint -save -e64*/
        if (0 != connect(socketFD, (struct sockaddr*)&si_host, sizeof(si_host)))
        {
            if (ENETUNREACH == errno)
            {
                mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p, ip='%s', port=%d) ERROR: Network not reachable (%d)",
                        __FUNCTION__, pContext, ipAddr, port, errno);

                close(socketFD);
                return MSPIN_ERROR_NETWORK_NOT_REACHABLE;
            }
            else if (ECONNREFUSED == errno)
            {
                mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p, ip='%s', port=%d) ERROR: Connection refused (%d)",
                        __FUNCTION__, pContext, ipAddr, port, errno);
                close(socketFD);
                return MSPIN_ERROR_CONNECTION_REFUSED;
            }
            else
            {
                mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p, ip='%s', port=%d) ERROR: Could not establish connection with '%s'(%d)",
                        __FUNCTION__, pContext, ipAddr, port, strerror(errno), errno);
                close(socketFD);
                return MSPIN_ERROR_CONNECTION_START;
            }
        }
        /*lint -restore*/
        //Release connection handles
        mspin_conn_releaseHandle(&(pContext->pConnectionHandle));

        //Create connection handle
        pConnectionHandle = mspin_conn_createHandle();
        if (!pConnectionHandle)
        {
            mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p, ip='%s', port=%d) ERROR: Failed to create connection handle",
                    __FUNCTION__, pContext, ipAddr, port);
            close(socketFD);
            return MSPIN_ERROR_MEM_ALLOC;
        }

        pConnectionHandle->connectionType = MSPIN_CONNECTION_CLIENT_TCP;
        pConnectionHandle->result = 0; //success
        pConnectionHandle->port = port;
        pContext->pConnectionHandle = pConnectionHandle;

        result = mspin_tcp_createNewConnection(socketFD, inet_addr((const char*)ipAddr));
        if (MSPIN_SUCCESS != result)
        {
            mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p, ip='%s', port=%d) ERROR: Failed to create connection",
                    __FUNCTION__, pContext, ipAddr, port);

            close(socketFD);
            return result;
        }

#ifdef MSPIN_TCP_USE_MESSAGE_QUEUE

        //Create message queue for incoming data
        if (pTCPHandle->mesgQueue < 0)
        {
            struct mq_attr attr;
            mqd_t queue = -1;
            mspin_log_printLn(eMspinVerbosityDebug, "%s(ctx=%p, ip='%s', port=%d) open read message queue",
                    __FUNCTION__, pContext, ipAddr, port);

            //Set message queue parameter
            attr.mq_curmsgs = 0;
            attr.mq_flags = 0;
            attr.mq_maxmsg = MSPIN_TCP_READ_MAX_NUMBER_MESSAGES;
            attr.mq_msgsize = MSPIN_TCP_READ_BUFFER_SIZE;

            queue = mq_open(MSPIN_TCP_MESSAGE_QUEUE, O_RDWR|O_CREAT, S_IRWXU, &attr);
            if (queue < 0)
            {
                mspin_log_printLn(eMspinVerbosityError,
                        "%s(ctx=%p, ip='%s', port=%d) ERROR: Failed to open read message queue with rc=%d (errno=%s)",
                        __FUNCTION__, pContext, ipAddr, port, queue, strerror(errno));

                //Cleanup before returning
                close(pTCPHandle->socketFD);
                pTCPHandle->socketFD = -1;

                free(pTCPHandle->ipAddr);
                pTCPHandle->ipAddr = NULL;

                free(pTCPHandle);
                pContext->pConnectionHandle->pTCPHandle = NULL;

                return MSPIN_ERROR_CONNECTION_START;
            }
            else
            {
                mspin_log_printLn(eMspinVerbosityDebug,
                        "%s(ctx=%p, ip='%s', port=%d) read message queue %d opened",
                        __FUNCTION__, pContext, ipAddr, port, queue);

                pTCPHandle->mesgQueue = queue;
            }
        }

        //Start reader thread
        mspin_tcp_startTCPReaderThread(pContext->pConnectionHandle->pTCPHandle);

#endif //MSPIN_TCP_USE_MESSAGE_QUEUE
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p, ip='%s', port=%d) ERROR: Context is NULL",
                __FUNCTION__, pContext, ipAddr, port);
        return MSPIN_ERROR_NULL_HANDLE;
    }

    mspin_log_printLn(eMspinVerbosityDebug, "%s(ctx=%p, ip='%s', port=%d) connection established",
            __FUNCTION__, pContext, ipAddr, port);

    return MSPIN_SUCCESS;
}

MSPIN_ERROR mspin_tcp_disconnectServer(mspin_context_t* pContext)
{
    S32 rc = 0;
    mspin_connectionHandle_t *pConnectionHandle = NULL;
    in_addr_t ipAddr = 0;
    char hostString[INET_ADDRSTRLEN] = "";

    mspin_log_printLn(eMspinVerbosityDebug, "%s(ctx=%p) entered", __FUNCTION__, pContext);

    if (pContext)
    {
        pConnectionHandle = pContext->pConnectionHandle;
        if (!pConnectionHandle)
        {
            mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p) ERROR: Connection handle is NULL",
                    __FUNCTION__, pContext);
            return MSPIN_ERROR_NULL_HANDLE;
        }

#ifdef MSPIN_TCP_USE_MESSAGE_QUEUE
        void* status = 0;
        //Terminate reader thread
        pTCPHandle->quitReader = TRUE;

        //Join reader thread
        if (pTCPHandle->threadID > 0)
        {
            (void)pthread_cancel(pTCPHandle->threadID);
            (void)pthread_join(pTCPHandle->threadID, &status);
            pTCPHandle->threadID = 0;
        }

        mspin_log_printLn(eMspinVerbosityDebug, "%s(ctx=%p) thread joined for address=%s:%d",
                __FUNCTION__, pContext, pTCPHandle->ipAddr, pTCPHandle->port);
#else
    MSPIN_UNUSED(rc);
#endif //MSPIN_TCP_USE_MESSAGE_QUEUE

        ipAddr = mspin_tcp_getIPAddr(pConnectionHandle->connectionID);
        inet_ntop(AF_INET, &(ipAddr), hostString, INET_ADDRSTRLEN);

        //Close socket
        mspin_tcp_closeConnection(pConnectionHandle->connectionID);

        mspin_log_printLn(eMspinVerbosityDebug, "%s(ctx=%p) socket=%d closed for address=%s:%d",
                __FUNCTION__, pContext, pConnectionHandle->connectionID, hostString, pConnectionHandle->port);
        pConnectionHandle->connectionID = -1;

#ifdef MSPIN_TCP_USE_MESSAGE_QUEUE
        //Close TCP message queue
        rc = mq_close(pTCPHandle->mesgQueue);
        if (0 == rc)
        {
            rc = mq_unlink(MSPIN_TCP_MESSAGE_QUEUE);

            if (0 != rc )
            {
                mspin_log_printLn(eMspinVerbosityError,
                        "%s(ctx=%p) ERROR: Failed to unlink message queue with rc=%d (errno: '%s')",
                        __FUNCTION__, pContext, rc, strerror(errno));
            }
            else
            {
                mspin_log_printLn(eMspinVerbosityDebug, "%s(ctx=%p) message queue %d closed for address=%s:%d",
                        __FUNCTION__, pContext, pTCPHandle->mesgQueue, pTCPHandle->ipAddr, pTCPHandle->port);
                pTCPHandle->mesgQueue = -1;
            }
        }
#endif //MSPIN_TCP_USE_MESSAGE_QUEUE

        mspin_log_printLn(eMspinVerbosityDebug, "%s(ctx=%p) connection closed",
                __FUNCTION__, pContext);

        return MSPIN_SUCCESS;
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p) ERROR: Context is NULL",
                __FUNCTION__, pContext);
        return MSPIN_ERROR_NULL_HANDLE;
    }
}

